<?php

namespace App\Console\Commands;

use App\Models\Package;
use App\Models\Purchase;
use App\Models\User;
use App\Models\UserLedger;
use Carbon\Carbon;
use Illuminate\Console\Command;

class CapitalReturnCommand extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'capital:return';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Return capital to users for expired packages with capital_back enabled';

    /**
     * Execute the console command.
     *
     * @return int
     */
    public function handle()
    {
        $this->info('Starting capital return process for packages with capital_back enabled...');

        // Get all purchases where validity has expired
        // Check both 'active' and 'inactive' statuses because AdminController marks expired as 'inactive'
        // Get all purchases with active or inactive status
        // We fetch all and filter by date in PHP because 'validity' is a varchar column
        // and SQL string comparison might be inaccurate for dates.
        $allPurchases = Purchase::whereIn('status', ['active', 'inactive'])
            ->with('package', 'user')
            ->get();

        $expiredPurchases = $allPurchases->filter(function ($purchase) {
            try {
                // Parse the validity date string. 
                // Assuming format might vary, so we try standard parsing.
                // If it's just a date string, Carbon::parse handles it well.
                $validityDate = Carbon::parse($purchase->validity);
                return $validityDate->isPast();
            } catch (\Exception $e) {
                // If date parsing fails, log it and skip
                \Illuminate\Support\Facades\Log::error("Failed to parse validity date for purchase #{$purchase->id}: {$purchase->validity}");
                return false;
            }
        });

        $processedCount = 0;

        foreach ($expiredPurchases as $purchase) {
            $package = Package::find($purchase->package_id);
            
            // Skip if package doesn't exist or doesn't have capital return enabled
            if (!$package || !$package->capital_back) {
                \Illuminate\Support\Facades\Log::info("Skipping purchase #{$purchase->id}: Package not found or capital_back disabled.");
                // If capital_back is No, mark as inactive (if not already inactive)
                if ($purchase->status == 'active') {
                    $purchase->status = 'inactive';
                    $purchase->save();
                }
                continue;
            }

            $user = User::find($purchase->user_id);
            
            if (!$user) {
                \Illuminate\Support\Facades\Log::info("Skipping purchase #{$purchase->id}: User not found.");
                // Mark as inactive if user not found
                if ($purchase->status == 'active') {
                    $purchase->status = 'inactive';
                    $purchase->save();
                }
                continue;
            }

            // Check if capital has already been returned for this purchase
            // Prevent double processing by checking ledger entries
            $capitalAlreadyReturned = UserLedger::where('user_id', $user->id)
                ->where('reason', 'capital_return')
                ->where('amount', $purchase->amount)
                ->where('perticulation', 'like', '%Capital Return - Package Expired%')
                ->whereDate('created_at', '>=', Carbon::parse($purchase->validity)->subDays(1))
                ->exists();

            if ($capitalAlreadyReturned) {
                $this->info("Capital already returned for purchase #{$purchase->id}, skipping...");
                \Illuminate\Support\Facades\Log::info("Skipping purchase #{$purchase->id}: Capital already returned.");
                // Mark as inactive if still active
                if ($purchase->status == 'active') {
                    $purchase->status = 'inactive';
                    $purchase->save();
                }
                continue;
            }

            // Return the CAPITAL (original investment amount), NOT the profit
            // When capital_back = Yes, we return the capital amount that user invested
            $capitalAmount = $purchase->amount;
            
            // Credit user balance with capital
            $user->deposit_wallet += $capitalAmount;
            $user->save();

            // Create ledger entry for capital return
            UserLedger::create([
                'user_id' => $user->id,
                'reason' => 'capital_return',
                'perticulation' => 'Capital Return - Package Expired (Capital Returned)',
                'amount' => $capitalAmount,
                'credit' => $capitalAmount,
                'status' => 'approved',
                'step' => 'first',
                'date' => Carbon::now()->format('d-m-Y H:i'),
            ]);

            // Mark purchase as inactive (already processed)
            $purchase->status = 'inactive';
            $purchase->save();

            $processedCount++;
            $this->info("Returned ৳{$capitalAmount} capital to user #{$user->id}");
            \Illuminate\Support\Facades\Log::info("Processed purchase #{$purchase->id}: Returned {$capitalAmount} to user {$user->id}");
        }

        $this->info("Capital return process completed. Processed {$processedCount} packages.");

        return 0;
    }
}
