<?php

namespace App\Http\Controllers;

use App\Models\UserCheckIn;
use App\Models\UserLedger;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class CheckInController extends Controller
{
    public function showCheckInPage()
    {
        $user = Auth::user();
        $rewards = [1, 2, 3, 4, 5, 6, 7];

        $today = Carbon::today();
        $registered = Carbon::parse($user->created_at)->startOfDay();
        $daysSinceRegister = $registered->diffInDays($today);
        $startOfUserWeek = $registered->copy()->addDays(floor($daysSinceRegister / 7) * 7);
        $nextCheckInTime = Carbon::tomorrow()->startOfDay()->toDateTimeString();

        $checkIns = UserCheckIn::where('user_id', $user->id)
            ->where('created_at', '>=', $startOfUserWeek)
            ->orderBy('created_at', 'asc')
            ->get();

        $claimedDays = [];

        foreach ($checkIns as $checkIn) {
            $dayIndex = $checkIn->created_at->diffInDays($startOfUserWeek);
            $claimedDays[$dayIndex] = true;
        }

        return view('app.main.check_in', compact('rewards', 'claimedDays', 'nextCheckInTime'));
    }

    public function checkIn(Request $request)
    {
        $user = Auth::user();
        $rewards = [1, 2, 3, 4, 5, 6, 7];

        $today = Carbon::today();
        $registered = Carbon::parse($user->created_at)->startOfDay();
        $daysSinceRegister = $registered->diffInDays($today);
        $startOfUserWeek = $registered->copy()->addDays(floor($daysSinceRegister / 7) * 7);
        $dayIndex = $today->diffInDays($startOfUserWeek);

        if ($dayIndex < 0 || $dayIndex > 6) {
            return response()->json([
                'message' => 'Check-in not available for today.',
                'claimed' => true
            ], 400);
        }

        $alreadyClaimed = UserCheckIn::where('user_id', $user->id)
            ->whereDate('created_at', $today)
            ->exists();

        if ($alreadyClaimed) {
            return response()->json([
                'message' => 'You have already checked in today.',
                'claimed' => true
            ], 400);
        }

        if ($request->input('day') - 1 !== $dayIndex) {
            return response()->json([
                'message' => 'You can only check in for today.',
                'claimed' => true
            ], 400);
        }

        $weeklyCheckIns = UserCheckIn::where('user_id', $user->id)
            ->where('created_at', '>=', $startOfUserWeek)
            ->orderBy('created_at', 'asc')
            ->get();

        $streak = true;
        for ($i = 0; $i <= $dayIndex; $i++) {
            $expectedDate = $startOfUserWeek->copy()->addDays($i)->toDateString();
            $hasCheckIn = $weeklyCheckIns->where('created_at', '>=', $expectedDate)
                ->where('created_at', '<', Carbon::parse($expectedDate)->addDay())
                ->count();

            if (!$hasCheckIn && $i != $dayIndex) {
                $streak = false;
                break;
            }
        }

        $rewardAmount = $rewards[$dayIndex];

        $user->increment('balance', $rewardAmount);
        $user->increment('total_income', $rewardAmount);
        $user->increment('today_income', $rewardAmount);

        UserCheckIn::create([
            'user_id' => $user->id,
            'reward' => $rewardAmount,
            'streak' => $streak ? 1 : 0,
        ]);

        UserLedger::create([
            'user_id' => $user->id,
            'reason' => 'check_in_reward',
            'perticulation' => 'Daily check-in reward',
            'amount' => $rewardAmount,
            'debit' => 0,
            'credit' => $rewardAmount,
            'status' => 'approved',
            'date' => Carbon::now()->format('Y-m-d H:i:s'),
            'claimed' => 1,
        ]);

        return response()->json([
            'message' => 'Check-in successful!',
            'balance' => $user->balance,
            'claimed' => false
        ]);
    }
}
