<?php

namespace App\Http\Controllers;

use App\Models\Deposit;
use App\Models\User;
use App\Models\PaymentMethod;
use App\Models\Withdrawal; 
use Exception;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;

class WatchPayController extends Controller
{
    // WatchPay API URL
    protected $baseUrl = 'https://api.watchglb.com';

    /**
     * 
     */
    public function payNow(array $field_data)
    {
        $amount = $field_data['amount'];
        $mchId = $field_data['mch_id'];
        $merchantKey = $field_data['merchant_key']; 
        $payType = $field_data['pay_type']; 
        
        $merchantOrderNo = rand(100000000, 900000000) . '2';
        $notifyUrl = route('watchpay.deposit.callback');
        $pageUrl = route('dashboard');
        $orderDate = date('Y-m-d H:i:s');

        $params = [
            'version' => '1.0',
            'mch_id' => $mchId,
            'notify_url' => $notifyUrl,
            'page_url' => $pageUrl,
            'mch_order_no' => $merchantOrderNo,
            'pay_type' => $payType,
            'trade_amount' => $amount,
            'order_date' => $orderDate,
            'goods_name' => $field_data['method_name'] ?? 'Deposit',
            'sign_type' => 'MD5',
        ];

        
        $params['sign'] = $this->_generateSign($params, $merchantKey);

        try {
            $response = Http::asForm()->post($this->baseUrl . '/pay/web', $params);
            
            if ($response->successful()) {
                $data = $response->json();
                
                if (isset($data['respCode']) && $data['respCode'] == 'SUCCESS' && isset($data['payInfo'])) {
                    
                    
                    $model = new Deposit();
                    $model->user_id = auth()->user()->id;
                    $model->method_name = $field_data['method_name'];
                    $model->order_id = $merchantOrderNo;
                    $model->transaction_id = $data['orderNo'] ?? null;
                    $model->amount = $amount;
                    $model->final_amount = $amount;
                    $model->date = $orderDate;
                    $model->status = 'drop'; 
                    $model->save();
                    
                    Log::info('WatchPay Deposit: Order Created', ['params' => $params, 'response' => $data]);

                    return redirect()->away($data['payInfo']);
                } else {
                    return back()->with('error', 'Gateway Error: ' . ($data['tradeMsg'] ?? 'Unknown error'));
                }
            } else {
                return back()->with('error', 'Gateway Connection Error.');
            }
        } catch (Exception $e) {
            Log::error('WatchPay Exception: ' . $e->getMessage());
            return back()->with('error', 'An unexpected error occurred.');
        }
    }

    /**
     * cllback/IPN)
     */
    public function ipn(Request $request)
    {
        Log::info('WatchPay IPN Received: ' . json_encode($request->all()));

        $data = $request->all();
        $sign = $data['sign'] ?? '';
        
        
        $mchId = $data['mchId'] ?? ($data['mch_id'] ?? ($data['merNo'] ?? null));

        if (!$mchId) {
            Log::error('WatchPay IPN Error: Merchant ID missing.');
            return 'fail';
        }

        
        $paymentMethod = PaymentMethod::where('mch_id', $mchId)->first();
        if (!$paymentMethod) {
            Log::error('WatchPay IPN Error: Merchant ID '.$mchId.' not found in Database.');
            return 'fail';
        }

        $merchantKey = $paymentMethod->merchant_key;

        
        if ($this->_verifySign($data, $merchantKey, $sign)) {
            
            // tradeResult = 1 )
            $tradeResult = $data['tradeResult'] ?? null;
            $mchOrderNo = $data['mchOrderNo'] ?? ($data['mch_order_no'] ?? null);

            if ($tradeResult == '1' && $mchOrderNo) {
                $deposit = Deposit::where('order_id', $mchOrderNo)
                    ->where('status', 'drop')->first();

                if ($deposit) {
                    $deposit->status = 'approved';
                    $deposit->save();

                    $user = User::find($deposit->user_id);
                    if ($user) {
                        $user->balance += $deposit->amount;
                        $user->save();
                        Log::info('WatchPay Success: Balance added to User '.$user->id);
                    }
                } else {
                    Log::warning('WatchPay IPN: Order ID '.$mchOrderNo.' not found or already processed.');
                }
                return 'success'; 
            } else {
                Log::warning('WatchPay IPN: Payment failed or Order No missing. Result: '.$tradeResult);
            }
        } else {
            Log::error('WatchPay IPN Error: Signature Mismatch.');
        }

        return 'fail';
    }

    
    /**
     * Withdrawal Request (Payout)
     * Ei function-ti admin panel theke taka pathanor jonno bebohar kora hoy.
     */
    public function payoutNow($field_data)
    {
        $mchId = $field_data['mch_id'];
        $withdrawKey = $field_data['secret_key']; 
        
        // Amount-ke integer-e convert kora hochhe (Documentation onujayi)
        $amount = (int) $field_data['amount'];
        $orderId = rand(100000000, 900000000) . '3';
        $applyDate = date('Y-m-d H:i:s');

        // Bank Code Mapping: 
        // Apnar deya specific wallet codes ekhane set kora hoyeche.
        $incomingCode = strtoupper((string)$field_data['bank_code']);
        
        if ($incomingCode == 'SF221' || str_contains($incomingCode, 'Bkash')) {
            $bankCode = 'BDT25000f062';
        } elseif ($incomingCode == 'SF222' || str_contains($incomingCode, 'Nagad')) {
            $bankCode = 'BDT25000f062';
        } else {
            // Jodi onno kono code thake, tobe seti pathano hobe
            $bankCode = $field_data['bank_code'];
        }

        // API Parameters Setup
        $params = [
            'mch_id'          => (string) $mchId,
            'mch_transferId'  => (string) $orderId,
            'transfer_amount' => (string) $amount,
            'apply_date'      => (string) $applyDate,
            'bank_code'       => (string) $bankCode, 
            'receive_name'    => (string) $field_data['holderName'],
            'receive_account' => (string) $field_data['account'],
            'sign_type'       => 'MD5'
        ];

        // Signature Generate (Secret Key bebohar kore)
        $params['sign'] = $this->_generateSign($params, $withdrawKey);

        try {
            // Request pathano hochhe asForm() bebohar kore (application/x-www-form-urlencoded)
            $response = \Illuminate\Support\Facades\Http::asForm()->post($this->baseUrl . '/pay/transfer', $params);
            
            if ($response->successful()) {
                $data = $response->json();

                if (isset($data['respCode']) && $data['respCode'] == 'SUCCESS') {
                    return [
                        'status'     => true,
                        'trx_id'     => $data['tradeNo'] ?? null,
                        'merOrderNo' => $data['merTransferId'] ?? $orderId,
                        'message'    => 'success',
                        'data'       => $data
                    ];
                } else {
                    return [
                        'status'  => false,
                        'message' => $data['errorMsg'] ?? 'Payout failed'
                    ];
                }
            }
            return ['status' => false, 'message' => 'Gateway Connection Error'];
        } catch (\Exception $e) {
            return ['status' => false, 'message' => 'Exception: ' . $e->getMessage()];
        }
    }

    /**
     * WatchPay Signature Verification
     * Verifies the callback signature by regenerating it and comparing.
     */
    private function _verifySign(array $data, string $secret, string $receivedSign): bool
    {
        $generatedSign = $this->_generateSign($data, $secret);
        return strtolower($generatedSign) === strtolower($receivedSign);
    }

    /**
     * WatchPay Signature Generation
     */
    private function _generateSign(array $params, string $secret): string
    {
        // sign, sign_type, ebong signType signature-e thakbe na
        unset($params['sign'], $params['sign_type'], $params['signType']);

        // Khali values bad deya hoyeche
        $params = array_filter($params, function($value) {
            return $value !== null && $value !== '';
        });

        // ASCII key onujayi sort kora
        ksort($params);

        $signStr = "";
        foreach ($params as $key => $value) {
            $signStr .= $key . "=" . $value . "&";
        }

        // Sheshe secret key jog kore MD5 hash kora hochhe
        $signSource = $signStr . "key=" . $secret;
        
        return md5($signSource);
    }
}