<?php

namespace App\Http\Controllers\admin;

use App\Http\Controllers\Controller;
use App\Models\JobApplication;
use App\Models\JobPosition;
use App\Models\User;
use App\Models\UserLedger;
use Illuminate\Http\Request;

class JobController extends Controller
{
    // --- Job Positions CRUD ---
    public function index()
    {
        $positions = JobPosition::orderBy('display_order', 'asc')->get();
        return view('admin.pages.job.index', compact('positions'));
    }

    public function moveUp($id)
    {
        $position = JobPosition::findOrFail($id);
        if ($position->display_order === null) {
            $maxOrder = JobPosition::max('display_order') ?? 0;
            $position->display_order = $maxOrder + 1;
            $position->save();
        }
        $previous = JobPosition::where('display_order', '<', $position->display_order)
            ->orderBy('display_order', 'desc')->first();
        if ($previous) {
            $temp = $position->display_order;
            $position->display_order = $previous->display_order;
            $previous->display_order = $temp;
            $position->save();
            $previous->save();
        }
        return redirect()->route('admin.job.index')->with('success', 'Position order updated.');
    }

    public function moveDown($id)
    {
        $position = JobPosition::findOrFail($id);
        if ($position->display_order === null) {
            $maxOrder = JobPosition::max('display_order') ?? 0;
            $position->display_order = $maxOrder + 1;
            $position->save();
        }
        $next = JobPosition::where('display_order', '>', $position->display_order)
            ->orderBy('display_order', 'asc')->first();
        if ($next) {
            $temp = $position->display_order;
            $position->display_order = $next->display_order;
            $next->display_order = $temp;
            $position->save();
            $next->save();
        }
        return redirect()->route('admin.job.index')->with('success', 'Position order updated.');
    }

    public function create()
    {
        return view('admin.pages.job.create');
    }

    public function store(Request $request)
    {
        $request->validate([
            'title' => 'required|string|max:255',
            'salary' => 'required|numeric',
            'require_l1' => 'required|integer',
            'status' => 'required|in:active,inactive',
        ]);
        JobPosition::create(array_merge(
            $request->only(['title', 'salary', 'require_l1', 'status']),
            ['display_order' => (JobPosition::max('display_order') ?? 0) + 1]
        ));
        return redirect()->route('admin.job.index')->with('success', 'Job Position created successfully.');
    }

    public function edit($id)
    {
        $position = JobPosition::findOrFail($id);
        return view('admin.pages.job.edit', compact('position'));
    }

    public function update(Request $request, $id)
    {
        $position = JobPosition::findOrFail($id);
        $request->validate([
            'title' => 'required|string|max:255',
            'salary' => 'required|numeric',
            'require_l1' => 'required|integer',
            'status' => 'required|in:active,inactive',
        ]);
        $position->update($request->only(['title', 'salary', 'require_l1', 'status']));
        return redirect()->route('admin.job.index')->with('success', 'Job Position updated successfully.');
    }

    public function destroy($id)
    {
        JobPosition::findOrFail($id)->delete();
        return redirect()->route('admin.job.index')->with('success', 'Job Position deleted.');
    }

    // --- Job Applications ---
    public function applications()
    {
        $applications = JobApplication::with(['user', 'position'])->latest()->paginate(50);
        return view('admin.pages.job.applications', compact('applications'));
    }

    // List only approved applications for monthly salary payment
    public function approvedApplications(Request $request)
    {
        $search = $request->search;
        $applications = JobApplication::with(['user', 'position'])
            ->where('status', 'approved')
            ->when($search, function($query) use ($search) {
                $query->whereHas('user', function($q) use ($search) {
                    $q->where('username', 'like', "%$search%")
                      ->orWhere('name', 'like', "%$search%");
                });
            })
            ->latest('paid_at')
            ->paginate(50);

        return view('admin.pages.job.approved_applications', compact('applications', 'search'));
    }

    public function approveApplication($id)
    {
        $application = JobApplication::findOrFail($id);
        
        // Mark any other active (approved) application for this user as 'superseded'
        JobApplication::where('user_id', $application->user_id)
            ->where('status', 'approved')
            ->where('id', '!=', $application->id)
            ->update(['status' => 'rejected']);

        $application->status = 'approved';
        $application->save();
        return back()->with('success', 'Application approved. Other active positions for this user have been deactivated.');
    }

    public function rejectApplication($id)
    {
        $application = JobApplication::findOrFail($id);
        $application->status = 'rejected';
        $application->save();
        return back()->with('success', 'Application rejected.');
    }

    public function paySalary($id)
    {
        $application = JobApplication::findOrFail($id);
        if ($application->status != 'approved') {
            return back()->with('error', 'Only approved applications can be paid.');
        }

        $user = $application->user;
        $salary = $application->position->salary;

        $user->balance += $salary;
        $user->save();

        // We DO NOT change status to 'salary_paid' anymore, to keep it 'approved' for next month.
        $application->paid_at = now();
        $application->save();

        UserLedger::create([
            'user_id' => $user->id,
            'reason' => 'job_salary',
            'perticulation' => 'Monthly Salary for position: ' . $application->position->title,
            'amount' => $salary,
            'credit' => $salary,
            'status' => 'approved',
            'date' => now()->format('d-m-Y H:i'),
        ]);

        return back()->with('success', 'Salary paid successfully.');
    }
}
