<?php

namespace App\Http\Controllers\user;

use App\Http\Controllers\Controller;
use App\Models\JobApplication;
use App\Models\JobPosition;
use App\Models\Setting;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class JobController extends Controller
{
    public function index()
    {
        $positions = JobPosition::where('status', 'active')->get();
        $user = Auth::user();
        $setting = Setting::find(1);
        $minInvestment = $setting->min_referral_investment ?? 1500;

        // Calculate user's referral counts based on cumulative approved recharges
        $validL1 = User::where('ref_by', $user->ref_id)
            ->withSum(['deposits' => function ($query) {
                $query->where('status', 'approved');
            }], 'final_amount')
            ->get()
            ->filter(function($u) use ($minInvestment) {
                return $u->deposits_sum_final_amount >= $minInvestment;
            })->count();

        // Standard ABC count logic from previous context (usually Level A, B, C total)
        // For simplicity, let's assume we need to count total valid referrals across all levels if ABC is required
        // This is a common pattern in these types of apps.
        
        // Fetch all applications for history display
        $history = JobApplication::with('position')
            ->where('user_id', $user->id)
            ->latest()
            ->get();

        // For the "Apply" buttons, we only care about the latest status of each position
        $user_applications = [];
        foreach ($history->reverse() as $app) {
            $user_applications[$app->job_position_id] = $app->status;
        }

        return view('app.main.job.index', compact('positions', 'user', 'validL1', 'user_applications', 'minInvestment', 'history'));
    }

    public function apply(Request $request)
    {
        $user = Auth::user();
        $position = JobPosition::findOrFail($request->position_id);
        $setting = Setting::find(1);
        $minInvestment = $setting->min_referral_investment ?? 1500;

        // Check eligibility again on submission
        $validL1 = User::where('ref_by', $user->ref_id)
            ->withSum(['deposits' => function ($query) {
                $query->where('status', 'approved');
            }], 'final_amount')
            ->get()
            ->filter(function($u) use ($minInvestment) {
                return $u->deposits_sum_final_amount >= $minInvestment;
            })->count();

        if ($validL1 < $position->require_l1) {
            return response()->json(['status' => 'error', 'message' => 'Insufficient Level A referrals.']);
        }

        // Check if already applied and NOT rejected
        // If rejected, they CAN apply again.
        $exists = JobApplication::where('user_id', $user->id)
            ->where('job_position_id', $position->id)
            ->whereIn('status', ['pending', 'approved']) 
            ->exists();
            
        if ($exists) {
            return response()->json(['status' => 'error', 'message' => 'You have an active or pending application for this position.']);
        }

        JobApplication::create([
            'user_id' => $user->id,
            'job_position_id' => $position->id,
            'status' => 'pending'
        ]);

        return response()->json(['status' => 'success', 'message' => 'Application submitted successfully. Wait for admin approval.']);
    }
}
