<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Lucky Wheel</title>
<link rel="icon" type="image/png" href="/static/img/images.png">
    
    <meta name="csrf-token" content="{{ csrf_token() }}">

    <link href="https://fonts.googleapis.com/css2?family=Poppins:wght@400;500;600;700&display=swap" rel="stylesheet">
    <script src="https://cdn.tailwindcss.com"></script>

    <script src="https://cdnjs.cloudflare.com/ajax/libs/tone/14.7.77/Tone.js"></script>

    <style>
        body { font-family: 'Poppins', sans-serif; background-color: #4863A0; }
        .spin-button:active { transform: scale(0.95); }
        @keyframes fadeInScaleUp {
            from { opacity: 0; transform: scale(0.8); }
            to { opacity: 1; transform: scale(1); }
        }
        .result-animation { animation: fadeInScaleUp 0.5s ease-out forwards; }
    </style>
</head>
<body>

<div class="flex items-center justify-center min-h-screen p-4">
    <div id="mainContainer" class="relative w-full max-w-sm mx-auto bg-black/20 backdrop-blur-xl rounded-3xl shadow-2xl p-6 md:p-8 text-center text-white border border-white/30">
        
        <div class="absolute top-4 left-4 z-30">
            <button onclick="window.history.back()" class="w-10 h-10 rounded-full bg-white/20 hover:bg-white/30 flex items-center justify-center transition-colors" aria-label="Go back">
                <svg xmlns="http://www.w3.org/2000/svg" class="h-6 w-6 text-white" fill="none" viewBox="0 0 24 24" stroke="currentColor" stroke-width="2">
                    <path stroke-linecap="round" stroke-linejoin="round" d="M15 19l-7-7 7-7" />
                </svg>
            </button>
        </div>

        <h1 class="text-3xl font-bold mb-2">Lucky Wheel</h1>
        <p class="text-gray-300 mb-6">Spin to win exciting prizes!</p>

        <div class="relative w-64 h-64 md:w-72 md:h-72 mx-auto mb-8">
            <div class="absolute inset-0 pointer-events-none">
                <div id="wheelContainer" class="absolute top-0 left-0 w-full h-full opacity-0 transition-opacity duration-500">
                    <canvas id="wheelCanvas" width="500" height="500" class="w-full h-full"></canvas>
                </div>
                
                <div id="wheelLoader" class="absolute inset-0 flex items-center justify-center">
                    <svg class="animate-spin h-12 w-12 text-white" xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24">
                        <circle class="opacity-25" cx="12" cy="12" r="10" stroke="currentColor" stroke-width="4"></circle>
                        <path class="opacity-75" fill="currentColor" d="M4 12a8 8 0 018-8V0C5.373 0 0 5.373 0 12h4zm2 5.291A7.962 7.962 0 014 12H0c0 3.042 1.135 5.824 3 7.938l3-2.647z"></path>
                    </svg>
                </div>

                <div class="absolute w-10 h-10 top-1/2 -right-2 -translate-y-1/2 flex items-center justify-center z-10">
                    <div class="w-0 h-0 border-t-[15px] border-t-transparent border-b-[15px] border-b-transparent border-l-[25px] border-l-amber-400 shadow-lg"></div>
                </div>
            </div>

            <div class="absolute inset-0 flex items-center justify-center z-20">
                <button id="spinBtn" class="spin-button w-16 h-16 md:w-20 md-h-20 text-white text-xl font-bold rounded-full border-4 border-white/80 shadow-lg flex items-center justify-center cursor-pointer transition-colors duration-300 bg-gray-600">
                    SPIN
                </button>
            </div>
        </div>

        <div class="h-12 flex items-center justify-center mb-4">
            <p id="resultText" class="text-xl font-semibold">&nbsp;</p>
        </div>
        
        <div class="bg-white/10 rounded-lg p-3 mb-6">
            <p class="text-lg font-semibold">Spins Remaining: <span id="spinsCount" class="bg-purple-700 px-3 py-1 rounded-full">0</span></p>
        </div>

        <div class="bg-black/20 border border-white/20 rounded-lg p-4 text-left min-h-[96px]">
            <h3 class="font-bold text-lg mb-2 text-center">How to get spins?</h3>
            <p class="text-sm text-gray-300 text-center">
                You get a free spin each time you purchase a package, or when a referred member makes a purchase.
            </p>
        </div>
    </div>
</div>

<script>
    document.addEventListener('DOMContentLoaded', function () {
        const API_URL_STATUS = '{{ route("user.wheel.status") }}';
        const API_URL_SUBMIT = '{{ route("user.wheel.submit") }}';
        
        const canvas = document.getElementById('wheelCanvas');
        const wheelContainer = document.getElementById('wheelContainer');
        const spinBtn = document.getElementById('spinBtn');
        const resultText = document.getElementById('resultText');
        const spinsCountEl = document.getElementById('spinsCount');
        
        let prizes = [];
        let spins = 0;
        let isSpinning = false;
        let currentRotation = 0;
        const ctx = canvas.getContext('2d');
        let isAudioInitialized = false;
        let winSynth;

        function initAudio() {
            if (isAudioInitialized) return;
            winSynth = new Tone.Synth({ oscillator: { type: 'triangle' }, envelope: { attack: 0.005, decay: 0.1, sustain: 0.3, release: 1 } }).toDestination();
            isAudioInitialized = true;
        }

        async function initializeWheel() {
            try {
                const response = await fetchAPI(API_URL_STATUS);
                prizes = response.data.prizes;
                spins = response.data.spins;
                drawWheel();
                updateSpinsDisplay();
                document.getElementById('wheelLoader').style.display = 'none';
                wheelContainer.classList.remove('opacity-0');
            } catch (error) {
                resultText.textContent = 'Error loading wheel.';
                console.error('Initialization Error:', error);
            }
        }

        function drawWheel() {
            if (!prizes || prizes.length === 0) return;
            const prizeCount = prizes.length;
            const anglePerPrize = (2 * Math.PI) / prizeCount;
            const canvasSize = canvas.width;
            const center = canvasSize / 2;
            const radius = center - 10;

            ctx.clearRect(0, 0, canvasSize, canvasSize);
            ctx.font = `bold ${canvasSize * 0.08}px 'Poppins', sans-serif`;
            ctx.textAlign = 'center';
            ctx.textBaseline = 'middle';

            prizes.forEach((prize, i) => {
                const startAngle = (i * anglePerPrize) - (anglePerPrize / 2);
                const endAngle = startAngle + anglePerPrize;
                
                ctx.beginPath();
                ctx.moveTo(center, center);
                ctx.arc(center, center, radius, startAngle, endAngle);
                ctx.closePath();
                ctx.fillStyle = prize.color;
                ctx.fill();
                ctx.strokeStyle = 'rgba(255, 255, 255, 0.5)';
                ctx.lineWidth = 4;
                ctx.stroke();

                ctx.save();
                ctx.fillStyle = 'white';
                const textAngle = i * anglePerPrize;
                ctx.translate(center, center);
                ctx.rotate(textAngle);
                ctx.fillText(prize.number, radius * 0.65, 0);
                ctx.restore();
            });
        }

        function updateSpinsDisplay() {
            spinsCountEl.textContent = spins;
            spinBtn.disabled = isSpinning;
            if (spins > 0 && !isSpinning) {
                spinBtn.classList.remove('bg-gray-600', 'cursor-not-allowed');
                spinBtn.classList.add('bg-green-500');
            } else {
                spinBtn.classList.remove('bg-green-500');
                spinBtn.classList.add('bg-gray-600', 'cursor-not-allowed');
            }
        }

        async function executeSpin() {
            if (isSpinning || spins <= 0) {
                if(spins <= 0 && !isSpinning) {
                    resultText.textContent = "You have no spins left.";
                    resultText.classList.add('result-animation');
                }
                return;
            }
            isSpinning = true;
            updateSpinsDisplay();
            resultText.innerHTML = "&nbsp;";
            resultText.classList.remove('result-animation');

            try {
                const response = await fetchAPI(API_URL_SUBMIT, 'POST');
                const { winningPrize, spinsLeft } = response.data;
                spins = spinsLeft;
                const winningIndex = prizes.findIndex(p => p.number == winningPrize.number);
                
                if (winningIndex === -1) throw new Error("Prize not found.");

                const degreesPerPrize = 360 / prizes.length;
                const spinDuration = 7000;
                
                const randomRotations = (Math.floor(Math.random() * 4) + 8) * 360;
                const targetAngle = 360 - (winningIndex * degreesPerPrize) - (degreesPerPrize / 2);

                
                let newTargetRotation = Math.ceil(currentRotation / 360) * 360;
                newTargetRotation += randomRotations + targetAngle;
                
                currentRotation = newTargetRotation;
                
                wheelContainer.style.transition = `transform ${spinDuration}ms cubic-bezier(0.25, 1, 0.5, 1)`;
                wheelContainer.style.transform = `rotate(${currentRotation}deg)`;

                setTimeout(() => finishSpin(winningPrize), spinDuration);

            } catch (error) {
                resultText.textContent = error.message || 'Spin failed.';
                isSpinning = false;
                updateSpinsDisplay();
            }
        }

        function finishSpin(prize) {
            isSpinning = false;
            updateSpinsDisplay();
            resultText.textContent = prize.number > 0 ? `Congratulations! You won ${prize.number}!` : `Better luck next time!`;
            resultText.classList.add('result-animation');

            if (isAudioInitialized) {
                const now = Tone.now();
                winSynth.triggerAttackRelease('C5', '8n', now);
                winSynth.triggerAttackRelease('G5', '8n', now + 0.2);
            }
        }
        
        async function fetchAPI(url, method = 'GET', body = null) {
            const csrfToken = document.querySelector('meta[name="csrf-token"]').getAttribute('content');
            const headers = { 'Content-Type': 'application/json', 'Accept': 'application/json', 'X-CSRF-TOKEN': csrfToken };
            const options = { method, headers, body: body ? JSON.stringify(body) : null };
            const response = await fetch(url, options);
            const data = await response.json();
            if (!response.ok || !data.success) throw new Error(data.message || 'API error.');
            return data;
        }

        spinBtn.addEventListener('click', async () => {
            if (!isAudioInitialized) {
                try {
                    await Tone.start();
                    initAudio();
                } catch (e) { console.error("Audio context error", e); }
            }
            executeSpin();
        });
        
        initializeWheel();
    });
</script>

</body>
</html>

